<?php

require_once __DIR__ . '/../models/Payroll.php';
require_once __DIR__ . '/../models/Employee.php';
require_once __DIR__ . '/../helpers/AuthHelper.php';

class PayrollsController {
    private $payroll;
    private $employee;
    
    public function __construct() {
        $this->payroll = new Payroll();
        $this->employee = new Employee();
    }
    
    /**
     * Listar folhas de pagamento
     */
    public function index() {
        AuthHelper::check();
        
        // Filtros
        $filters = [
            'employee_id' => $_GET['employee_id'] ?? '',
            'reference_month' => $_GET['month'] ?? date('n'),
            'reference_year' => $_GET['year'] ?? date('Y'),
            'status' => $_GET['status'] ?? ''
        ];
        
        // Buscar dados
        $payrolls = $this->payroll->findAll($filters);
        $employees = $this->employee->getAll();
        $statistics = $this->payroll->getStatistics($filters['reference_month'], $filters['reference_year']);
        
        // Carregar view
        require_once __DIR__ . '/../views/payrolls/index.php';
    }
    
    /**
     * Exibir formulário de criação
     */
    public function create() {
        AuthHelper::check();
        
        $employees = $this->employee->getAll();
        $selectedEmployee = $_GET['employee_id'] ?? '';
        $selectedMonth = $_GET['month'] ?? date('n');
        $selectedYear = $_GET['year'] ?? date('Y');
        
        // Se colaborador foi selecionado, calcular automaticamente
        $calculatedData = null;
        if ($selectedEmployee && $selectedMonth && $selectedYear) {
            try {
                $calculatedData = $this->payroll->calculatePayroll($selectedEmployee, $selectedMonth, $selectedYear);
            } catch (Exception $e) {
                $_SESSION['error'] = $e->getMessage();
            }
        }
        
        require_once __DIR__ . '/../views/payrolls/create.php';
    }
    
    /**
     * Salvar nova folha de pagamento
     */
    public function store() {
        AuthHelper::check();
        
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            header('Location: ' . BASE_URL . '/payrolls');
            exit;
        }
        
        try {
            // Validar dados
            $data = [
                'employee_id' => $_POST['employee_id'] ?? '',
                'reference_month' => $_POST['reference_month'] ?? '',
                'reference_year' => $_POST['reference_year'] ?? '',
                'base_salary' => floatval($_POST['base_salary'] ?? 0),
                'overtime_hours' => floatval($_POST['overtime_hours'] ?? 0),
                'overtime_rate' => floatval($_POST['overtime_rate'] ?? 0),
                'bonus' => floatval($_POST['bonus'] ?? 0),
                'allowances' => floatval($_POST['allowances'] ?? 0),
                'inss_discount' => floatval($_POST['inss_discount'] ?? 0),
                'irrf_discount' => floatval($_POST['irrf_discount'] ?? 0),
                'other_discounts' => floatval($_POST['other_discounts'] ?? 0),
                'net_salary' => floatval($_POST['net_salary'] ?? 0),
                'worked_hours' => floatval($_POST['worked_hours'] ?? 0),
                'status' => $_POST['status'] ?? 'draft',
                'notes' => $_POST['notes'] ?? ''
            ];
            
            // Validar
            $errors = $this->payroll->validate($data);
            
            // Verificar se já existe folha para o período
            if ($this->payroll->existsForPeriod($data['employee_id'], $data['reference_month'], $data['reference_year'])) {
                $errors[] = 'Já existe uma folha de pagamento para este colaborador no período selecionado';
            }
            
            if (!empty($errors)) {
                $_SESSION['error'] = implode('<br>', $errors);
                $_SESSION['old_data'] = $data;
                header('Location: ' . BASE_URL . '/payrolls/create');
                exit;
            }
            
            // Salvar
            if ($this->payroll->create($data)) {
                $_SESSION['success'] = 'Folha de pagamento criada com sucesso!';
                header('Location: ' . BASE_URL . '/payrolls');
            } else {
                $_SESSION['error'] = 'Erro ao criar folha de pagamento';
                $_SESSION['old_data'] = $data;
                header('Location: ' . BASE_URL . '/payrolls/create');
            }
            
        } catch (Exception $e) {
            $_SESSION['error'] = 'Erro: ' . $e->getMessage();
            $_SESSION['old_data'] = $_POST;
            header('Location: ' . BASE_URL . '/payrolls/create');
        }
        
        exit;
    }
    
    /**
     * Exibir detalhes da folha de pagamento
     */
    public function show($id) {
        AuthHelper::check();
        
        $payroll = $this->payroll->findById($id);
        
        if (!$payroll) {
            $_SESSION['error'] = 'Folha de pagamento não encontrada';
            header('Location: ' . BASE_URL . '/payrolls');
            exit;
        }
        
        require_once __DIR__ . '/../views/payrolls/show.php';
    }
    
    /**
     * Exibir formulário de edição
     */
    public function edit($id) {
        AuthHelper::check();
        
        $payroll = $this->payroll->findById($id);
        
        if (!$payroll) {
            $_SESSION['error'] = 'Folha de pagamento não encontrada';
            header('Location: ' . BASE_URL . '/payrolls');
            exit;
        }
        
        $employees = $this->employee->getAll();
        
        require_once __DIR__ . '/../views/payrolls/edit.php';
    }
    
    /**
     * Atualizar folha de pagamento
     */
    public function update($id) {
        AuthHelper::check();
        
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            header('Location: ' . BASE_URL . '/payrolls');
            exit;
        }
        
        try {
            // Verificar se existe
            $existingPayroll = $this->payroll->findById($id);
            if (!$existingPayroll) {
                $_SESSION['error'] = 'Folha de pagamento não encontrada';
                header('Location: ' . BASE_URL . '/payrolls');
                exit;
            }
            
            // Validar dados
            $data = [
                'base_salary' => floatval($_POST['base_salary'] ?? 0),
                'overtime_hours' => floatval($_POST['overtime_hours'] ?? 0),
                'overtime_rate' => floatval($_POST['overtime_rate'] ?? 0),
                'bonus' => floatval($_POST['bonus'] ?? 0),
                'allowances' => floatval($_POST['allowances'] ?? 0),
                'inss_discount' => floatval($_POST['inss_discount'] ?? 0),
                'irrf_discount' => floatval($_POST['irrf_discount'] ?? 0),
                'other_discounts' => floatval($_POST['other_discounts'] ?? 0),
                'net_salary' => floatval($_POST['net_salary'] ?? 0),
                'worked_hours' => floatval($_POST['worked_hours'] ?? 0),
                'status' => $_POST['status'] ?? 'draft',
                'notes' => $_POST['notes'] ?? ''
            ];
            
            // Validar
            $errors = [];
            if ($data['base_salary'] < 0) $errors[] = 'Salário base deve ser maior ou igual a zero';
            if ($data['net_salary'] < 0) $errors[] = 'Salário líquido não pode ser negativo';
            
            if (!empty($errors)) {
                $_SESSION['error'] = implode('<br>', $errors);
                header('Location: ' . BASE_URL . '/payrolls/edit/' . $id);
                exit;
            }
            
            // Atualizar
            if ($this->payroll->update($id, $data)) {
                $_SESSION['success'] = 'Folha de pagamento atualizada com sucesso!';
                header('Location: ' . BASE_URL . '/payrolls/show/' . $id);
            } else {
                $_SESSION['error'] = 'Erro ao atualizar folha de pagamento';
                header('Location: ' . BASE_URL . '/payrolls/edit/' . $id);
            }
            
        } catch (Exception $e) {
            $_SESSION['error'] = 'Erro: ' . $e->getMessage();
            header('Location: ' . BASE_URL . '/payrolls/edit/' . $id);
        }
        
        exit;
    }
    
    /**
     * Excluir folha de pagamento
     */
    public function destroy($id) {
        AuthHelper::check();
        
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            header('Location: ' . BASE_URL . '/payrolls');
            exit;
        }
        
        try {
            $payroll = $this->payroll->findById($id);
            
            if (!$payroll) {
                $_SESSION['error'] = 'Folha de pagamento não encontrada';
                header('Location: ' . BASE_URL . '/payrolls');
                exit;
            }
            
            // Não permitir exclusão de folhas aprovadas
            if ($payroll['status'] === 'approved') {
                $_SESSION['error'] = 'Não é possível excluir folhas de pagamento aprovadas';
                header('Location: ' . BASE_URL . '/payrolls');
                exit;
            }
            
            if ($this->payroll->delete($id)) {
                $_SESSION['success'] = 'Folha de pagamento excluída com sucesso!';
            } else {
                $_SESSION['error'] = 'Erro ao excluir folha de pagamento';
            }
            
        } catch (Exception $e) {
            $_SESSION['error'] = 'Erro: ' . $e->getMessage();
        }
        
        header('Location: ' . BASE_URL . '/payrolls');
        exit;
    }
    
    /**
     * Calcular folha automaticamente via AJAX
     */
    public function calculate() {
        AuthHelper::check();
        
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            http_response_code(405);
            echo json_encode(['error' => 'Método não permitido']);
            exit;
        }
        
        try {
            $employeeId = $_POST['employee_id'] ?? '';
            $month = $_POST['month'] ?? '';
            $year = $_POST['year'] ?? '';
            
            if (!$employeeId || !$month || !$year) {
                throw new Exception('Parâmetros obrigatórios não informados');
            }
            
            $data = $this->payroll->calculatePayroll($employeeId, $month, $year);
            
            header('Content-Type: application/json');
            echo json_encode([
                'success' => true,
                'data' => $data
            ]);
            
        } catch (Exception $e) {
            http_response_code(400);
            header('Content-Type: application/json');
            echo json_encode([
                'success' => false,
                'error' => $e->getMessage()
            ]);
        }
        
        exit;
    }
    
    /**
     * Gerar folhas em lote
     */
    public function generateBatch() {
        AuthHelper::check();
        
        if ($_SERVER['REQUEST_METHOD'] === 'GET') {
            // Exibir formulário
            $employees = $this->employee->getAll();
            require_once __DIR__ . '/../views/payrolls/batch.php';
            return;
        }
        
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            header('Location: ' . BASE_URL . '/payrolls');
            exit;
        }
        
        try {
            $month = $_POST['month'] ?? '';
            $year = $_POST['year'] ?? '';
            $employeeIds = $_POST['employee_ids'] ?? [];
            
            if (!$month || !$year) {
                $_SESSION['error'] = 'Mês e ano são obrigatórios';
                header('Location: ' . BASE_URL . '/payrolls/generate-batch');
                exit;
            }
            
            // Gerar folhas
            $results = $this->payroll->generateBatchPayroll($month, $year, $employeeIds);
            
            // Contar sucessos e erros
            $successCount = count(array_filter($results, function($r) { return $r['success']; }));
            $errorCount = count($results) - $successCount;
            
            if ($successCount > 0) {
                $_SESSION['success'] = "$successCount folha(s) de pagamento criada(s) com sucesso!";
            }
            
            if ($errorCount > 0) {
                $errorMessages = [];
                foreach ($results as $result) {
                    if (!$result['success']) {
                        $errorMessages[] = $result['employee_name'] . ': ' . $result['message'];
                    }
                }
                $_SESSION['error'] = "$errorCount erro(s):<br>" . implode('<br>', $errorMessages);
            }
            
        } catch (Exception $e) {
            $_SESSION['error'] = 'Erro: ' . $e->getMessage();
        }
        
        header('Location: ' . BASE_URL . '/payrolls');
        exit;
    }
    
    /**
     * Aprovar folha de pagamento
     */
    public function approve($id) {
        AuthHelper::check();
        
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            header('Location: ' . BASE_URL . '/payrolls');
            exit;
        }
        
        try {
            $payroll = $this->payroll->findById($id);
            
            if (!$payroll) {
                $_SESSION['error'] = 'Folha de pagamento não encontrada';
                header('Location: ' . BASE_URL . '/payrolls');
                exit;
            }
            
            if ($payroll['status'] === 'approved') {
                $_SESSION['error'] = 'Esta folha já está aprovada';
                header('Location: ' . BASE_URL . '/payrolls/show/' . $id);
                exit;
            }
            
            // Aprovar
            $data = ['status' => 'approved'] + $payroll;
            
            if ($this->payroll->update($id, $data)) {
                $_SESSION['success'] = 'Folha de pagamento aprovada com sucesso!';
            } else {
                $_SESSION['error'] = 'Erro ao aprovar folha de pagamento';
            }
            
        } catch (Exception $e) {
            $_SESSION['error'] = 'Erro: ' . $e->getMessage();
        }
        
        header('Location: ' . BASE_URL . '/payrolls/show/' . $id);
        exit;
    }
    
    /**
     * Exportar relatório em CSV
     */
    public function exportCsv() {
        AuthHelper::check();
        
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            header('Location: ' . BASE_URL . '/payrolls');
            exit;
        }
        
        try {
            $filters = [
                'employee_id' => $_POST['employee_id'] ?? '',
                'reference_month' => $_POST['month'] ?? '',
                'reference_year' => $_POST['year'] ?? '',
                'status' => $_POST['status'] ?? ''
            ];
            
            $payrolls = $this->payroll->findAll($filters);
            
            if (empty($payrolls)) {
                $_SESSION['error'] = 'Nenhum registro encontrado para exportação';
                header('Location: ' . BASE_URL . '/payrolls');
                exit;
            }
            
            // Gerar CSV
            $filename = 'folhas_pagamento_' . date('Y-m-d_H-i-s') . '.csv';
            
            header('Content-Type: text/csv; charset=utf-8');
            header('Content-Disposition: attachment; filename="' . $filename . '"');
            header('Cache-Control: must-revalidate, post-check=0, pre-check=0');
            header('Expires: 0');
            
            $output = fopen('php://output', 'w');
            
            // BOM para UTF-8
            fprintf($output, chr(0xEF).chr(0xBB).chr(0xBF));
            
            // Cabeçalho
            fputcsv($output, [
                'ID',
                'Colaborador',
                'CPF',
                'Cargo',
                'Mês/Ano',
                'Salário Base',
                'Horas Extras',
                'Bônus',
                'Auxílios',
                'Desconto INSS',
                'Desconto IRRF',
                'Outros Descontos',
                'Salário Líquido',
                'Horas Trabalhadas',
                'Status'
            ], ';');
            
            // Dados
            foreach ($payrolls as $payroll) {
                fputcsv($output, [
                    $payroll['id'],
                    $payroll['employee_name'],
                    $payroll['cpf'],
                    $payroll['position'],
                    Payroll::getMonthName($payroll['reference_month']) . '/' . $payroll['reference_year'],
                    number_format($payroll['base_salary'], 2, ',', '.'),
                    $payroll['overtime_hours'],
                    number_format($payroll['bonus'], 2, ',', '.'),
                    number_format($payroll['allowances'], 2, ',', '.'),
                    number_format($payroll['inss_discount'], 2, ',', '.'),
                    number_format($payroll['irrf_discount'], 2, ',', '.'),
                    number_format($payroll['other_discounts'], 2, ',', '.'),
                    number_format($payroll['net_salary'], 2, ',', '.'),
                    $payroll['worked_hours'],
                    ucfirst($payroll['status'])
                ], ';');
            }
            
            fclose($output);
            
        } catch (Exception $e) {
            $_SESSION['error'] = 'Erro ao exportar: ' . $e->getMessage();
            header('Location: ' . BASE_URL . '/payrolls');
        }
        
        exit;
    }
}