<?php
require_once __DIR__ . '/../config/config.php';

/**
 * Serviço para envio de mensagens WhatsApp
 * Suporta múltiplos provedores: Twilio, UltraMsg, etc.
 */
class WhatsAppService {
    private $pdo;
    private $provider;
    private $config;
    
    public function __construct() {
        try {
            $this->pdo = new PDO('mysql:host=' . DB_HOST . ';dbname=' . DB_NAME, DB_USER, DB_PASS);
            $this->pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
            $this->loadConfig();
        } catch (PDOException $e) {
            error_log("Erro na conexão WhatsApp Service: " . $e->getMessage());
            throw new Exception("Erro na conexão com banco de dados");
        }
    }
    
    /**
     * Carrega configurações do sistema
     */
    private function loadConfig() {
        $sql = "SELECT chave, valor FROM configuracoes_sistema WHERE chave LIKE 'whatsapp_%'";
        $stmt = $this->pdo->query($sql);
        $configs = $stmt->fetchAll(PDO::FETCH_KEY_PAIR);
        
        $this->config = [
            'ativo' => ($configs['whatsapp_ativo'] ?? '0') === '1',
            'provider' => $configs['whatsapp_api_provider'] ?? 'twilio',
            'template' => $configs['whatsapp_template_compromisso'] ?? 'Lembrete: {titulo} em {data} às {hora}',
            'antecedencia_horas' => intval($configs['whatsapp_antecedencia_horas'] ?? 2),
            'horario_inicio' => $configs['whatsapp_horario_inicio'] ?? '08:00',
            'horario_fim' => $configs['whatsapp_horario_fim'] ?? '20:00'
        ];
        
        $this->provider = $this->config['provider'];
    }
    
    /**
     * Verifica se o WhatsApp está ativo
     */
    public function isActive() {
        return $this->config['ativo'];
    }
    
    /**
     * Envia mensagem WhatsApp
     */
    public function enviarMensagem($telefone, $mensagem, $idAgenda = null) {
        if (!$this->isActive()) {
            throw new Exception('Serviço WhatsApp não está ativo');
        }
        
        // Registra a tentativa no banco
        $notificacaoId = $this->registrarNotificacao($idAgenda, $telefone, $mensagem);
        
        try {
            $resultado = $this->enviarPorProvider($telefone, $mensagem);
            
            if ($resultado['sucesso']) {
                $this->atualizarNotificacao($notificacaoId, 'Enviada', $resultado['resposta']);
                return true;
            } else {
                $this->atualizarNotificacao($notificacaoId, 'Erro', $resultado['resposta'], $resultado['erro']);
                return false;
            }
        } catch (Exception $e) {
            $this->atualizarNotificacao($notificacaoId, 'Falhou', null, $e->getMessage());
            throw $e;
        }
    }
    
    /**
     * Envia mensagem baseado no provider configurado
     */
    private function enviarPorProvider($telefone, $mensagem) {
        switch ($this->provider) {
            case 'twilio':
                return $this->enviarViaTwilio($telefone, $mensagem);
            case 'ultramsg':
                return $this->enviarViaUltraMsg($telefone, $mensagem);
            default:
                throw new Exception('Provider WhatsApp não suportado: ' . $this->provider);
        }
    }
    
    /**
     * Envia via Twilio WhatsApp API
     */
    private function enviarViaTwilio($telefone, $mensagem) {
        // Busca credenciais do usuário admin
        $sql = "SELECT whatsapp_api_key, whatsapp_account_sid, whatsapp_phone_number FROM usuarios WHERE nivel_acesso = 'Admin' AND whatsapp_api_key IS NOT NULL LIMIT 1";
        $stmt = $this->pdo->query($sql);
        $credenciais = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$credenciais || !$credenciais['whatsapp_api_key']) {
            throw new Exception('Credenciais Twilio não configuradas');
        }
        
        $accountSid = $credenciais['whatsapp_account_sid'];
        $authToken = $credenciais['whatsapp_api_key'];
        $fromNumber = 'whatsapp:' . $credenciais['whatsapp_phone_number'];
        $toNumber = 'whatsapp:' . $this->formatarTelefone($telefone);
        
        $url = "https://api.twilio.com/2010-04-01/Accounts/{$accountSid}/Messages.json";
        
        $data = [
            'From' => $fromNumber,
            'To' => $toNumber,
            'Body' => $mensagem
        ];
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($data));
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_USERPWD, $accountSid . ':' . $authToken);
        curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/x-www-form-urlencoded']);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        $responseData = json_decode($response, true);
        
        if ($httpCode === 201 && isset($responseData['sid'])) {
            return [
                'sucesso' => true,
                'resposta' => $response,
                'message_id' => $responseData['sid']
            ];
        } else {
            return [
                'sucesso' => false,
                'resposta' => $response,
                'erro' => $responseData['message'] ?? 'Erro desconhecido na API Twilio'
            ];
        }
    }
    
    /**
     * Envia via UltraMsg (alternativa mais barata)
     */
    private function enviarViaUltraMsg($telefone, $mensagem) {
        // Implementação para UltraMsg API
        // Esta é uma API mais simples e barata para WhatsApp
        throw new Exception('Provider UltraMsg ainda não implementado');
    }
    
    /**
     * Formata telefone para padrão internacional
     */
    private function formatarTelefone($telefone) {
        // Remove caracteres não numéricos
        $telefone = preg_replace('/[^0-9]/', '', $telefone);
        
        // Se não tem código do país, adiciona +55 (Brasil)
        if (strlen($telefone) === 11 && substr($telefone, 0, 1) === '9') {
            $telefone = '55' . $telefone;
        } elseif (strlen($telefone) === 10) {
            $telefone = '559' . $telefone;
        } elseif (strlen($telefone) < 13) {
            $telefone = '55' . $telefone;
        }
        
        return '+' . $telefone;
    }
    
    /**
     * Registra tentativa de notificação no banco
     */
    private function registrarNotificacao($idAgenda, $telefone, $mensagem) {
        $sql = "INSERT INTO notificacoes_whatsapp (id_agenda, telefone_destino, mensagem, status) VALUES (?, ?, ?, 'Pendente')";
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute([$idAgenda, $telefone, $mensagem]);
        return $this->pdo->lastInsertId();
    }
    
    /**
     * Atualiza status da notificação
     */
    private function atualizarNotificacao($id, $status, $resposta = null, $erro = null) {
        $sql = "UPDATE notificacoes_whatsapp SET status = ?, resposta_api = ?, erro_detalhes = ?, data_envio = NOW(), tentativas = tentativas + 1 WHERE id = ?";
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute([$status, $resposta, $erro, $id]);
    }
    
    /**
     * Processa mensagem com template
     */
    public function processarTemplate($dados) {
        $template = $this->config['template'];
        
        $substituicoes = [
            '{titulo}' => $dados['titulo'] ?? '',
            '{data}' => isset($dados['data_inicio']) ? date('d/m/Y', strtotime($dados['data_inicio'])) : '',
            '{hora}' => isset($dados['data_inicio']) ? date('H:i', strtotime($dados['data_inicio'])) : '',
            '{descricao}' => $dados['descricao'] ?? '',
            '{categoria}' => $dados['categoria'] ?? ''
        ];
        
        return str_replace(array_keys($substituicoes), array_values($substituicoes), $template);
    }
    
    /**
     * Busca compromissos que precisam de notificação
     */
    public function buscarCompromissosParaNotificar() {
        $antecedenciaHoras = $this->config['antecedencia_horas'];
        $horarioInicio = $this->config['horario_inicio'];
        $horarioFim = $this->config['horario_fim'];
        
        $sql = "
            SELECT a.*, e.nome_completo, e.telefone 
            FROM agendas a 
            LEFT JOIN eleitores e ON a.id_eleitor = e.id 
            WHERE a.whatsapp_notificacao = 1 
            AND a.notificacao_enviada = 0 
            AND a.data_inicio BETWEEN NOW() AND DATE_ADD(NOW(), INTERVAL ? HOUR)
            AND TIME(a.data_inicio) BETWEEN ? AND ?
            AND (e.telefone IS NOT NULL AND e.telefone != '')
        ";
        
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute([$antecedenciaHoras, $horarioInicio, $horarioFim]);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
    
    /**
     * Marca compromisso como notificado
     */
    public function marcarComoNotificado($idAgenda) {
        $sql = "UPDATE agendas SET notificacao_enviada = 1, data_notificacao = NOW() WHERE id = ?";
        $stmt = $this->pdo->prepare($sql);
        return $stmt->execute([$idAgenda]);
    }
    
    /**
     * Obtém estatísticas de notificações
     */
    public function getEstatisticas($dataInicio = null, $dataFim = null) {
        $where = "";
        $params = [];
        
        if ($dataInicio && $dataFim) {
            $where = "WHERE DATE(data_criacao) BETWEEN ? AND ?";
            $params = [$dataInicio, $dataFim];
        }
        
        $sql = "
            SELECT 
                status,
                COUNT(*) as total,
                DATE(data_criacao) as data
            FROM notificacoes_whatsapp 
            {$where}
            GROUP BY status, DATE(data_criacao)
            ORDER BY data DESC
        ";
        
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute($params);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
}
?>