<?php
/**
 * contas/db.php — Conexão compatível com $mysqli e $conexao
 * Baseado em src/config/db.php mas adaptado para mysqli
 */
declare(strict_types=1);

if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

date_default_timezone_set('America/Belem');

// Carrega variáveis de ambiente LOCAIS (dentro da pasta contas)
$envPath = __DIR__ . '/env.php';
if (file_exists($envPath)) {
    require_once $envPath;
}

// Definição de credenciais (prioridade para variáveis de ambiente carregadas)
$DB_HOST = getenv('DB_HOST') ?: '127.0.0.1';
$DB_PORT = getenv('DB_PORT') ? (int) getenv('DB_PORT') : 3306;
$DB_USER = getenv('DB_USER') ?: 'root';
$DB_PASS = getenv('DB_PASS') ?: '';
$DB_NAME = getenv('DB_NAME_CONTAS') ?: (getenv('DB_NAME') ?: 'DB_CONTAS');
$DB_SOCKET = getenv('DB_SOCKET') ?: null;

mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT);

$mysqli = null;
$lastEx = null;
$dsnTried = [];

try {
    if (!empty($DB_SOCKET)) {
        $dsnTried[] = "socket: $DB_SOCKET";
        $mysqli = new mysqli($DB_HOST, $DB_USER, $DB_PASS, $DB_NAME, $DB_PORT, $DB_SOCKET);
    } else {
        $dsnTried[] = "host: $DB_HOST:$DB_PORT";
        $mysqli = new mysqli($DB_HOST, $DB_USER, $DB_PASS, $DB_NAME, $DB_PORT);
    }
} catch (mysqli_sql_exception $e) {
    $lastEx = $e;
    // Tenta sockets comuns se falhar
    $sockets = [
        '/var/run/mysqld/mysqld.sock',
        '/tmp/mysql.sock',
        '/Applications/MAMP/tmp/mysql/mysql.sock',
        'C:/xampp/mysql/mysql.sock',
    ];

    foreach ($sockets as $sock) {
        try {
            $dsnTried[] = "socket: $sock";
            $mysqli = new mysqli('localhost', $DB_USER, $DB_PASS, $DB_NAME, 0, $sock);
            $lastEx = null;
            break;
        } catch (mysqli_sql_exception $e2) {
            $lastEx = $e2;
        }
    }
}

if ($lastEx) {
    http_response_code(500);
    $msg = "Erro de conexão ao MySQL (Contas).\n";
    $msg .= "Tentativas:\n - " . implode("\n - ", $dsnTried) . "\n";
    $msg .= "Detalhes: " . $lastEx->getMessage();
    die(nl2br(htmlspecialchars($msg)));
}

$mysqli->set_charset('utf8mb4');

// 1. Tenta criar a tabela se não existir (apenas para garantir)
try {
    $mysqli->query("CREATE TABLE IF NOT EXISTS transacoes (
        id INT NOT NULL AUTO_INCREMENT,
        data_transacao DATE NOT NULL,
        tipo ENUM('Receita','Despesa') NOT NULL,
        descricao VARCHAR(255) NOT NULL,
        valor DECIMAL(10,2) NOT NULL,
        forma_pagamento VARCHAR(100) NULL,
        comprovante_pagamento VARCHAR(255) NULL,
        entidade VARCHAR(255) NULL,
        origem VARCHAR(100) NOT NULL,
        status ENUM('pendente','pago','vencido') DEFAULT 'pendente',
        data_pagamento DATE NULL,
        PRIMARY KEY (id)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci");
} catch (Exception $e) {
    // Se a tabela já existe ou erro de permissão, segue o baile
}

// 2. AUTO-CORREÇÃO: Verifica colunas e faz ALTER se necessário
// Separado do bloco de criação para garantir que rode mesmo se a criação falhar/pular
try {
    // Verifica comprovante_pagamento
    try {
        $mysqli->query("SELECT comprovante_pagamento FROM transacoes LIMIT 1");
    } catch (mysqli_sql_exception $e) {
        $mysqli->query("ALTER TABLE transacoes ADD COLUMN comprovante_pagamento VARCHAR(255) NULL AFTER forma_pagamento");
        $mysqli->query("ALTER TABLE transacoes MODIFY COLUMN entidade VARCHAR(255) NULL");
    }

    // Verifica status e data_pagamento
    try {
        $mysqli->query("SELECT status FROM transacoes LIMIT 1");
    } catch (mysqli_sql_exception $e) {
        $mysqli->query("ALTER TABLE transacoes ADD COLUMN status ENUM('pendente','pago','vencido') DEFAULT 'pendente'");
        $mysqli->query("ALTER TABLE transacoes ADD COLUMN data_pagamento DATE NULL");
    }
} catch (Exception $e) {
    // Log de erro silencioso ou aviso
}

// Alias de compatibilidade
if (!isset($conexao) || !$conexao) {
    $conexao = $mysqli;
}

function db_conn()
{
    global $conexao, $mysqli;
    return $conexao ?: ($mysqli ?? null);
}
