<?php

require_once __DIR__ . '/../helpers/AuthHelper.php';
require_once __DIR__ . '/../config/config.php';
require_once __DIR__ . '/../models/Conta.php';

class ContasController
{
    private $contaModel;

    public function __construct()
    {
        if (session_status() === PHP_SESSION_NONE) {
            session_start();
        }
        $this->contaModel = new Conta();
    }

    public function index()
    {
        AuthHelper::check();

        $ano = isset($_GET['ano']) ? (int) $_GET['ano'] : date('Y');
        $mes = isset($_GET['mes']) ? (int) $_GET['mes'] : (int) date('m');

        $transacoes = $this->contaModel->getAll($ano, $mes);
        $resumo = $this->contaModel->getResumo($ano, $mes);

        $total_receitas = $resumo['total_receitas'] ?? 0;
        $total_despesas = $resumo['total_despesas'] ?? 0;
        $saldo_total = $total_receitas - $total_despesas;

        require_once __DIR__ . '/../views/contas/index.php';
    }

    public function store()
    {
        AuthHelper::check();

        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            header('Location: ' . BASE_URL . '/index.php?url=contas');
            exit;
        }

        try {
            $tipo_e_origem = explode(':', $_POST['tipo_origem']);
            $tipo = $tipo_e_origem[0] ?? '';
            $origem = $tipo_e_origem[1] ?? '';
            $data_transacao = $_POST['data_transacao'] ?? '';
            $descricao = trim($_POST['descricao'] ?? '');
            $valor_str = str_replace(',', '.', $_POST['valor'] ?? '0');
            $valor = (float) $valor_str;

            if (empty($data_transacao) || empty($tipo) || empty($descricao) || $valor <= 0) {
                throw new Exception("Preencha todos os campos obrigatórios.");
            }

            // Upload de comprovante
            $comprovante_nome = null;
            if (isset($_FILES['comprovante_pagamento']) && $_FILES['comprovante_pagamento']['error'] == UPLOAD_ERR_OK) {
                $comprovante_nome = $this->uploadArquivo($_FILES['comprovante_pagamento']);
            }

            $dados = [
                'data_transacao' => $data_transacao,
                'tipo' => $tipo,
                'origem' => $origem,
                'descricao' => $descricao,
                'valor' => $valor,
                'forma_pagamento' => $_POST['forma_pagamento'] ?? null,
                'entidade' => $_POST['entidade'] ?? null,
                'comprovante_pagamento' => $comprovante_nome
            ];

            $this->contaModel->create($dados);

            $_SESSION['success'] = 'Transação registrada com sucesso!';
        } catch (Exception $e) {
            $_SESSION['error'] = 'Erro ao registrar: ' . $e->getMessage();
        }

        // Redireciona mantendo o filtro de data se possível, ou para o mês atual
        $ano = date('Y', strtotime($data_transacao));
        $mes = date('m', strtotime($data_transacao));
        header("Location: " . BASE_URL . "/index.php?url=contas&ano=$ano&mes=$mes");
        exit;
    }

    public function delete()
    {
        AuthHelper::check();

        $id = $_POST['id'] ?? null;
        if (!$id) {
            $_SESSION['error'] = 'ID inválido.';
            header('Location: ' . BASE_URL . '/index.php?url=contas');
            exit;
        }

        try {
            // Recuperar info para deletar arquivo
            $transacao = $this->contaModel->findById($id);
            if ($transacao && $transacao['comprovante_pagamento']) {
                $filePath = __DIR__ . '/../../contas/uploads/comprovantes/' . $transacao['comprovante_pagamento'];
                if (file_exists($filePath)) {
                    @unlink($filePath);
                }
            }

            $this->contaModel->delete($id);
            $_SESSION['success'] = 'Transação excluída com sucesso!';
        } catch (Exception $e) {
            $_SESSION['error'] = 'Erro ao excluir: ' . $e->getMessage();
        }

        $ano = $_POST['ano'] ?? date('Y');
        $mes = $_POST['mes'] ?? date('m');
        header("Location: " . BASE_URL . "/index.php?url=contas&ano=$ano&mes=$mes");
        exit;
    }

    private function uploadArquivo($file)
    {
        // Usar o diretório existente para manter compatibilidade ou criar um novo em src/uploads?
        // Vamos manter o local original para não quebrar links antigos se houver migração
        $dir = __DIR__ . '/../../contas/uploads/comprovantes';
        if (!is_dir($dir)) {
            @mkdir($dir, 0777, true);
        }

        $nomeOriginal = preg_replace('/[^a-zA-Z0-9._-]/', '_', $file['name']);
        $ext = pathinfo($nomeOriginal, PATHINFO_EXTENSION);
        $alvo = uniqid('comp_', true) . ($ext ? '.' . strtolower($ext) : '');
        $destino = $dir . '/' . $alvo;

        if (move_uploaded_file($file['tmp_name'], $destino)) {
            return $alvo;
        }
        return null;
    }
}
