<?php
class Documento
{
    private $pdo;

    // Tipos de arquivo permitidos por categoria
    private $tiposPermitidos = [
        'documento' => ['pdf', 'doc', 'docx', 'txt', 'odt'],
        'imagem' => ['jpg', 'jpeg', 'png', 'gif', 'webp', 'svg'],
        'video' => ['mp4', 'avi', 'mov', 'wmv', 'flv', 'webm']
    ];

    // Tamanho máximo por tipo (em bytes)
    private $tamanhoMaximo = [
        'documento' => 10485760, // 10MB
        'imagem' => 5242880,     // 5MB
        'video' => 104857600     // 100MB
    ];

    public function __construct()
    {
        try {
            $this->pdo = new PDO('mysql:host=' . DB_HOST . ';dbname=' . DB_NAME, DB_USER, DB_PASS);
            $this->pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
            $this->ensureSchema();
        } catch (PDOException $e) {
            die("Erro na conexão com o banco de dados: " . $e->getMessage());
        }
    }

    private function ensureSchema()
    {
        $sql = "CREATE TABLE IF NOT EXISTS documentos (
            id INT NOT NULL AUTO_INCREMENT,
            id_usuario INT NOT NULL,
            titulo VARCHAR(255) NOT NULL,
            descricao TEXT NULL,
            nome_arquivo VARCHAR(255) NOT NULL,
            caminho_arquivo VARCHAR(255) NOT NULL,
            tipo_arquivo VARCHAR(100) NOT NULL,
            tamanho_arquivo INT NOT NULL,
            categoria ENUM('Ofício','Requerimento','Projetos','Instituto','Prestação de Contas','Diversos','Foto Publicitária','Vídeo Publicitário','Material de Campanha','Peça Gráfica') NOT NULL,
            data_upload TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci";
        $this->pdo->exec($sql);
    }

    public function create($data)
    {
        // Versão simplificada até as colunas serem criadas no banco
        $sql = "INSERT INTO documentos (id_usuario, titulo, nome_arquivo, caminho_arquivo, tipo_arquivo, tamanho_arquivo, categoria) VALUES (:id_usuario, :titulo, :nome_arquivo, :caminho_arquivo, :tipo_arquivo, :tamanho_arquivo, :categoria)";
        $stmt = $this->pdo->prepare($sql);

        // Filtrar apenas os dados que existem na tabela atual
        $filteredData = [
            ':id_usuario' => $data['id_usuario'],
            ':titulo' => $data['titulo'],
            ':nome_arquivo' => $data['nome_arquivo'],
            ':caminho_arquivo' => $data['caminho_arquivo'],
            ':tipo_arquivo' => $data['tipo_arquivo'],
            ':tamanho_arquivo' => $data['tamanho_arquivo'],
            ':categoria' => $data['categoria'] ?? 'Documento'
        ];

        return $stmt->execute($filteredData);
    }

    public function update($id, $data)
    {
        $fields = [];
        $params = [':id' => $id];

        // Campos sempre atualizáveis
        if (isset($data['titulo'])) {
            $fields[] = "titulo = :titulo";
            $params['titulo'] = $data['titulo'];
        }
        if (isset($data['descricao'])) {
            $fields[] = "descricao = :descricao";
            $params['descricao'] = $data['descricao'];
        }
        if (isset($data['categoria'])) {
            $fields[] = "categoria = :categoria";
            $params['categoria'] = $data['categoria'];
        }
        if (isset($data['data_upload'])) {
            $fields[] = "data_upload = :data_upload";
            $params['data_upload'] = $data['data_upload'];
        }

        // Campos de arquivo (opcionais na edição)
        if (isset($data['nome_arquivo'])) {
            $fields[] = "nome_arquivo = :nome_arquivo";
            $params['nome_arquivo'] = $data['nome_arquivo'];
        }
        if (isset($data['caminho_arquivo'])) {
            $fields[] = "caminho_arquivo = :caminho_arquivo";
            $params['caminho_arquivo'] = $data['caminho_arquivo'];
        }
        if (isset($data['tipo_arquivo'])) {
            $fields[] = "tipo_arquivo = :tipo_arquivo";
            $params['tipo_arquivo'] = $data['tipo_arquivo'];
        }
        if (isset($data['tamanho_arquivo'])) {
            $fields[] = "tamanho_arquivo = :tamanho_arquivo";
            $params['tamanho_arquivo'] = $data['tamanho_arquivo'];
        }
        if (isset($data['tipo_midia'])) {
            // Temporariamente comentado se a coluna não existir, mas idealmente deveria estar aqui
            // $fields[] = "tipo_midia = :tipo_midia";
            // $params['tipo_midia'] = $data['tipo_midia'];
        }
        if (isset($data['largura'])) {
            // $fields[] = "largura = :largura";
            // $params['largura'] = $data['largura'];
        }
        if (isset($data['altura'])) {
            // $fields[] = "altura = :altura";
            // $params['altura'] = $data['altura'];
        }

        if (empty($fields)) {
            return true; // Nada a atualizar
        }

        $sql = "UPDATE documentos SET " . implode(', ', $fields) . " WHERE id = :id";
        $stmt = $this->pdo->prepare($sql);
        return $stmt->execute($params);
    }

    public function find($categoria = null, $tipo_midia = null)
    {
        $sql = "SELECT d.*, u.nome as nome_usuario FROM documentos d JOIN usuarios u ON d.id_usuario = u.id WHERE 1=1";
        $params = [];

        if ($categoria && $categoria != '') {
            $sql .= " AND d.categoria = :categoria";
            $params['categoria'] = $categoria;
        }

        // Temporariamente comentado até a coluna ser criada no banco
        /*
        if ($tipo_midia && $tipo_midia != '') {
            $sql .= " AND d.tipo_midia = :tipo_midia";
            $params['tipo_midia'] = $tipo_midia;
        }
        */

        $sql .= " ORDER BY d.data_upload DESC";

        $stmt = $this->pdo->prepare($sql);
        $stmt->execute($params);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    /**
     * Valida se o arquivo é permitido para o tipo de mídia
     */
    public function validarArquivo($arquivo, $tipo_midia)
    {
        $extensao = strtolower(pathinfo($arquivo['name'], PATHINFO_EXTENSION));

        // Verifica se a extensão é permitida
        if (!in_array($extensao, $this->tiposPermitidos[$tipo_midia])) {
            return ['valido' => false, 'erro' => 'Tipo de arquivo não permitido para ' . $tipo_midia];
        }

        // Verifica o tamanho
        if ($arquivo['size'] > $this->tamanhoMaximo[$tipo_midia]) {
            $tamanhoMB = $this->tamanhoMaximo[$tipo_midia] / 1048576;
            return ['valido' => false, 'erro' => 'Arquivo muito grande. Máximo: ' . $tamanhoMB . 'MB'];
        }

        return ['valido' => true];
    }

    /**
     * Determina o tipo de mídia baseado na extensão do arquivo
     */
    public function determinarTipoMidia($nomeArquivo)
    {
        $extensao = strtolower(pathinfo($nomeArquivo, PATHINFO_EXTENSION));

        foreach ($this->tiposPermitidos as $tipo => $extensoes) {
            if (in_array($extensao, $extensoes)) {
                return $tipo;
            }
        }

        return 'documento'; // padrão
    }

    /**
     * Obtém informações de imagem (largura e altura)
     */
    public function obterInfoImagem($caminhoArquivo)
    {
        if (function_exists('getimagesize')) {
            $info = getimagesize($caminhoArquivo);
            if ($info) {
                return ['largura' => $info[0], 'altura' => $info[1]];
            }
        }
        return ['largura' => null, 'altura' => null];
    }

    /**
     * Busca documentos por título (versão simplificada)
     */
    public function buscar($termo, $tipo_midia = null)
    {
        // Versão simplificada usando apenas colunas existentes
        $sql = "SELECT d.*, u.nome as nome_usuario FROM documentos d 
                JOIN usuarios u ON d.id_usuario = u.id 
                WHERE d.titulo LIKE :termo";

        $params = ['termo' => '%' . $termo . '%'];

        // Comentado até a coluna tipo_midia ser criada
        /*
        if ($tipo_midia && $tipo_midia != '') {
            $sql .= " AND d.tipo_midia = :tipo_midia";
            $params['tipo_midia'] = $tipo_midia;
        }
        */

        $sql .= " ORDER BY d.data_upload DESC";

        $stmt = $this->pdo->prepare($sql);
        $stmt->execute($params);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    /**
     * Obtém estatísticas dos documentos
     */
    public function obterEstatisticas()
    {
        // Versão simplificada usando apenas colunas existentes
        $sql = "SELECT 
                    categoria,
                    COUNT(*) as total,
                    SUM(tamanho_arquivo) as tamanho_total
                FROM documentos 
                GROUP BY categoria
                ORDER BY categoria";

        $stmt = $this->pdo->prepare($sql);
        $stmt->execute();
        $resultados = $stmt->fetchAll(PDO::FETCH_ASSOC);

        // Calcular estatísticas gerais
        $stats = [
            'total' => 0,
            'imagens' => 0,
            'videos' => 0,
            'tamanho_total' => 0
        ];

        foreach ($resultados as $row) {
            $stats['total'] += $row['total'];
            $stats['tamanho_total'] += $row['tamanho_total'];

            // Categorizar por tipo baseado na categoria
            if (
                strpos(strtolower($row['categoria']), 'foto') !== false ||
                strpos(strtolower($row['categoria']), 'imagem') !== false
            ) {
                $stats['imagens'] += $row['total'];
            } elseif (strpos(strtolower($row['categoria']), 'video') !== false) {
                $stats['videos'] += $row['total'];
            }
        }

        return $stats;
    }

    /**
     * Busca um documento por ID
     */
    public function findById($id)
    {
        $sql = "SELECT d.*, u.nome as nome_usuario FROM documentos d 
                JOIN usuarios u ON d.id_usuario = u.id 
                WHERE d.id = :id";

        $stmt = $this->pdo->prepare($sql);
        $stmt->execute(['id' => $id]);
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }

    /**
     * Exclui um documento
     */
    public function delete($id)
    {
        // Primeiro busca o documento para obter o caminho do arquivo
        $documento = $this->findById($id);
        if (!$documento) {
            return false;
        }

        // Remove o arquivo físico
        if (file_exists($documento['caminho_arquivo'])) {
            unlink($documento['caminho_arquivo']);
        }

        // Remove do banco de dados
        $sql = "DELETE FROM documentos WHERE id = ?";
        $stmt = $this->pdo->prepare($sql);

        return $stmt->execute([$id]);
    }

    /**
     * Busca documentos com filtros avançados
     */
    public function buscarComFiltros($filtros)
    {
        $sql = "SELECT d.*, u.nome as nome_usuario FROM documentos d 
                JOIN usuarios u ON d.id_usuario = u.id WHERE 1=1";
        $params = [];

        // Filtro por busca (título ou descrição)
        if (!empty($filtros['busca'])) {
            $sql .= " AND (d.titulo LIKE :busca OR d.descricao LIKE :busca)";
            $params['busca'] = '%' . $filtros['busca'] . '%';
        }

        // Filtro por categoria
        if (!empty($filtros['categoria'])) {
            $sql .= " AND d.categoria = :categoria";
            $params['categoria'] = $filtros['categoria'];
        }

        // Filtro por data
        if (!empty($filtros['data_inicio'])) {
            $sql .= " AND DATE(d.data_upload) >= :data_inicio";
            $params['data_inicio'] = $filtros['data_inicio'];
        }

        if (!empty($filtros['data_fim'])) {
            $sql .= " AND DATE(d.data_upload) <= :data_fim";
            $params['data_fim'] = $filtros['data_fim'];
        }

        $sql .= " ORDER BY d.data_upload DESC";

        // Paginação
        if (isset($filtros['limit']) && isset($filtros['offset'])) {
            $sql .= " LIMIT :limit OFFSET :offset";
            $params['limit'] = $filtros['limit'];
            $params['offset'] = $filtros['offset'];
        }

        $stmt = $this->pdo->prepare($sql);

        // Bind dos parâmetros de paginação separadamente (PDO requirement)
        foreach ($params as $key => $value) {
            if ($key === 'limit' || $key === 'offset') {
                $stmt->bindValue(':' . $key, (int) $value, PDO::PARAM_INT);
            } else {
                $stmt->bindValue(':' . $key, $value);
            }
        }

        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    /**
     * Conta documentos com filtros (para paginação)
     */
    public function contarComFiltros($filtros)
    {
        $sql = "SELECT COUNT(*) as total FROM documentos d WHERE 1=1";
        $params = [];

        // Filtro por busca (título ou descrição)
        if (!empty($filtros['busca'])) {
            $sql .= " AND (d.titulo LIKE :busca OR d.descricao LIKE :busca)";
            $params['busca'] = '%' . $filtros['busca'] . '%';
        }

        // Filtro por categoria
        if (!empty($filtros['categoria'])) {
            $sql .= " AND d.categoria = :categoria";
            $params['categoria'] = $filtros['categoria'];
        }

        // Filtro por data
        if (!empty($filtros['data_inicio'])) {
            $sql .= " AND DATE(d.data_upload) >= :data_inicio";
            $params['data_inicio'] = $filtros['data_inicio'];
        }

        if (!empty($filtros['data_fim'])) {
            $sql .= " AND DATE(d.data_upload) <= :data_fim";
            $params['data_fim'] = $filtros['data_fim'];
        }

        $stmt = $this->pdo->prepare($sql);
        $stmt->execute($params);
        $result = $stmt->fetch(PDO::FETCH_ASSOC);

        return $result['total'];
    }
}