<?php
class Eleitor
{
    private $pdo;
    private $cols = [];

    public function __construct()
    {
        try {
            $this->pdo = new PDO('mysql:host=' . DB_HOST . ';dbname=' . DB_NAME, DB_USER, DB_PASS);
            $this->pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
            // Schema é conhecido, não precisamos detectar
            $this->cols['nome'] = 'nomecompleto';
            $this->cols['data_nascimento'] = 'datanascimento';
            $this->cols['telefone'] = 'telefone';
            $this->cols['email'] = 'email';
            $this->cols['endereco'] = 'endereco';
            $this->cols['bairro'] = 'bairro';
            $this->cols['cidade'] = 'cidade';
            $this->cols['uf'] = 'uf';
            $this->cols['cep'] = 'cep';
            $this->cols['sexo'] = 'sexo';
            $this->cols['tipo'] = 'tipo';
            $this->cols['observacao'] = 'observacao';
        } catch (PDOException $e) {
            die("Erro na conexão com o banco de dados: " . $e->getMessage());
        }
    }

    // detectColumns removido pois temos o schema fixo agora

    public function getAll($usuarioId = null)
    {
        $sql = "SELECT id, {$this->cols['nome']} AS nome_completo, {$this->cols['telefone']} AS telefone, {$this->cols['bairro']} AS bairro FROM eleitores";

        if ($usuarioId) {
            $sql .= " WHERE usuario_id = :usuario_id";
        }

        $sql .= " ORDER BY {$this->cols['nome']} ASC";

        $stmt = $this->pdo->prepare($sql);

        if ($usuarioId) {
            $stmt->bindValue(':usuario_id', $usuarioId, PDO::PARAM_INT);
        }

        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    /**
     * Insere um novo eleitor no banco de dados com os novos campos.
     */
    /**
     * Insere um novo eleitor no banco de dados com os novos campos.
     */
    public function create($dados)
    {
        // Garante que colunas extras sejam consideradas se existirem no banco
        // Mas para segurança e clareza, vamos listar explicitamente as colunas padrão + novas

        $sql = "INSERT INTO eleitores (
                    {$this->cols['nome']}, 
                    cpf,
                    usuario_id,
                    {$this->cols['data_nascimento']}, 
                    {$this->cols['telefone']}, 
                    {$this->cols['email']}, 
                    {$this->cols['endereco']}, 
                    {$this->cols['bairro']}, 
                    {$this->cols['cidade']}, 
                    {$this->cols['uf']}, 
                    {$this->cols['cep']}, 
                    {$this->cols['sexo']}, 
                    {$this->cols['tipo']}, 
                    demanda,
                    {$this->cols['observacao']}
                )
                VALUES (
                    :nome_completo, 
                    :cpf,
                    :usuario_id,
                    :data_nascimento, 
                    :telefone, 
                    :email, 
                    :endereco, 
                    :bairro, 
                    :cidade, 
                    :uf, 
                    :cep, 
                    :sexo, 
                    :tipo, 
                    :demanda,
                    :observacao
                )";

        $stmt = $this->pdo->prepare($sql);

        // Associa os valores do array $dados aos parâmetros da query
        $stmt->execute([
            ':nome_completo' => $dados['nome_completo'],
            ':cpf' => !empty($dados['cpf']) ? $dados['cpf'] : null,
            ':usuario_id' => $dados['usuario_id'] ?? null,
            ':data_nascimento' => !empty($dados['data_nascimento']) ? $dados['data_nascimento'] : null,
            ':telefone' => $dados['telefone'] ?? null,
            ':email' => $dados['email'] ?? null,
            ':endereco' => $dados['endereco'] ?? null,
            ':bairro' => $dados['bairro'] ?? null,
            ':cidade' => $dados['cidade'] ?? 'Belém',
            ':uf' => $dados['uf'] ?? 'PA',
            ':cep' => $dados['cep'] ?? null,
            ':sexo' => $dados['sexo'] ?? null,
            ':tipo' => $dados['tipo'],
            ':demanda' => $dados['demanda'] ?? null,
            ':observacao' => $dados['observacao'] ?? null,
        ]);

        return $this->pdo->lastInsertId();
    }

    /**
     * Busca eleitores por nome, telefone ou bairro
     */
    public function search($termo, $usuarioId = null)
    {
        $sql = "SELECT id, {$this->cols['nome']} AS nome_completo, {$this->cols['telefone']} AS telefone, {$this->cols['bairro']} AS bairro FROM eleitores 
                WHERE ({$this->cols['nome']} LIKE :termo 
                OR {$this->cols['telefone']} LIKE :termo 
                OR {$this->cols['bairro']} LIKE :termo)";

        if ($usuarioId) {
            $sql .= " AND usuario_id = :usuario_id";
        }

        $sql .= " ORDER BY {$this->cols['nome']} ASC";

        $stmt = $this->pdo->prepare($sql);
        $stmt->bindValue(':termo', '%' . $termo . '%');

        if ($usuarioId) {
            $stmt->bindValue(':usuario_id', $usuarioId, PDO::PARAM_INT);
        }

        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    /**
     * Busca um eleitor por ID
     */
    /**
     * Busca um eleitor por ID
     */
    public function findById($id)
    {
        $sql = "SELECT 
                    id, 
                    {$this->cols['nome']} AS nome_completo, 
                    {$this->cols['data_nascimento']} AS data_nascimento,
                    {$this->cols['telefone']} AS telefone,
                    {$this->cols['email']} AS email,
                    {$this->cols['endereco']} AS endereco,
                    {$this->cols['bairro']} AS bairro,
                    {$this->cols['cidade']} AS cidade,
                    {$this->cols['uf']} AS uf,
                    {$this->cols['cep']} AS cep,
                    {$this->cols['sexo']} AS sexo,
                    {$this->cols['tipo']} AS tipo,
                    demanda,
                    {$this->cols['observacao']} AS observacao,
                    cpf
                FROM eleitores WHERE id = :id";
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute([':id' => $id]);
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }

    /**
     * Atualiza os dados de um eleitor
     */
    public function update($id, $dados)
    {
        $sql = "UPDATE eleitores SET 
                {$this->cols['nome']} = :nome_completo,
                cpf = :cpf,
                {$this->cols['data_nascimento']} = :data_nascimento,
                {$this->cols['telefone']} = :telefone,
                {$this->cols['email']} = :email,
                {$this->cols['endereco']} = :endereco,
                {$this->cols['bairro']} = :bairro,
                {$this->cols['cidade']} = :cidade,
                {$this->cols['uf']} = :uf,
                {$this->cols['cep']} = :cep,
                {$this->cols['sexo']} = :sexo,
                {$this->cols['tipo']} = :tipo,
                demanda = :demanda,
                {$this->cols['observacao']} = :observacao
                WHERE id = :id";

        $stmt = $this->pdo->prepare($sql);
        return $stmt->execute([
            ':id' => $id,
            ':nome_completo' => $dados['nome_completo'],
            ':cpf' => !empty($dados['cpf']) ? $dados['cpf'] : null,
            ':data_nascimento' => !empty($dados['data_nascimento']) ? $dados['data_nascimento'] : null,
            ':telefone' => $dados['telefone'] ?? null,
            ':email' => $dados['email'] ?? null,
            ':endereco' => $dados['endereco'] ?? null,
            ':bairro' => $dados['bairro'] ?? null,
            ':cidade' => $dados['cidade'] ?? 'Belém',
            ':uf' => $dados['uf'] ?? 'PA',
            ':cep' => $dados['cep'] ?? null,
            ':sexo' => $dados['sexo'] ?? null,
            ':tipo' => $dados['tipo'],
            ':demanda' => $dados['demanda'] ?? null,
            ':observacao' => $dados['observacao'] ?? null,
        ]);
    }

    /**
     * Exclui um eleitor do banco de dados
     */
    public function delete($id)
    {
        try {
            // Primeiro, verifica se o eleitor existe
            $eleitor = $this->findById($id);
            if (!$eleitor) {
                return false;
            }

            // Exclui o eleitor
            $sql = "DELETE FROM eleitores WHERE id = :id";
            $stmt = $this->pdo->prepare($sql);
            return $stmt->execute([':id' => $id]);
        } catch (PDOException $e) {
            error_log("Erro ao excluir eleitor: " . $e->getMessage());
            return false;
        }
    }
}