<?php
declare(strict_types=1);

require_once __DIR__ . '/../helpers/AuthHelper.php';
require_once __DIR__ . '/../config/config.php';

// Carrega o model disponível (Employee OU Employees)
$employeeModelLoaded = false;
if (file_exists(__DIR__ . '/../models/Employee.php')) {
    require_once __DIR__ . '/../models/Employee.php';
    $employeeModelLoaded = true;
}
if (file_exists(__DIR__ . '/../models/Employees.php')) {
    require_once __DIR__ . '/../models/Employees.php';
    $employeeModelLoaded = true;
}
if (!$employeeModelLoaded) {
    throw new Exception('Model Employee/Employees não encontrado em /models');
}

class EmployeesController {
    private $employeeModel;

    public function __construct() {
        if (session_status() === PHP_SESSION_NONE) session_start();
        if (class_exists('Employee'))   { $this->employeeModel = new Employee();   }
        elseif (class_exists('Employees')) { $this->employeeModel = new Employees(); }
        else { throw new Exception('Classe de model Employee/Employees não encontrada.'); }
    }

    /* ======================== HELPERS ======================== */

    private function publicRoot(): string {
        if (!defined('APP_ROOT')) define('APP_ROOT', dirname(__DIR__, 2));
        return is_dir(APP_ROOT . '/public') ? (APP_ROOT . '/public') : APP_ROOT;
    }

    private function parseIniBytes(string $val): int {
        $val = trim($val);
        $last = strtolower(substr($val, -1));
        $num = (int)$val;
        switch ($last) {
            case 'g': $num *= 1024;
            case 'm': $num *= 1024;
            case 'k': $num *= 1024;
        }
        return $num;
    }

    private function brMoneyToDecimal(string $v): float {
        $v = preg_replace('/[^\d,\.]/', '', $v);
        $v = str_replace('.', '', $v);
        $v = str_replace(',', '.', $v);
        return (float)($v === '' ? 0 : $v);
    }

    /** Faz upload/remover foto. Retorna [photo_path|null, errors[]] */
    private function handlePhotoUpload(?string $currentPhotoPath, bool $removePhoto): array {
        $errors = [];
        $newPhotoPath = $currentPhotoPath;

        // Remoção sem novo upload
        if ($removePhoto && empty($_FILES['photo']['name'])) {
            if (!empty($currentPhotoPath)) {
                $old = $this->publicRoot() . '/' . ltrim($currentPhotoPath, '/');
                if (is_file($old)) @unlink($old);
            }
            return [null, $errors];
        }

        // Sem arquivo novo: mantém atual
        if (empty($_FILES['photo']['name'])) {
            return [$newPhotoPath, $errors];
        }

        // Validar upload
        $tmp  = $_FILES['photo']['tmp_name'] ?? '';
        $size = (int)($_FILES['photo']['size'] ?? 0);
        $name = (string)($_FILES['photo']['name'] ?? '');
        $ext  = strtolower(pathinfo($name, PATHINFO_EXTENSION));

        // Limites de tamanho (php.ini)
        $postMax    = $this->parseIniBytes(ini_get('post_max_size'));
        $uplMax     = $this->parseIniBytes(ini_get('upload_max_filesize'));
        $contentLen = (int)($_SERVER['CONTENT_LENGTH'] ?? 0);
        if ($contentLen > 0 && $contentLen > $postMax) {
            $errors[] = 'O tamanho total do formulário excede o limite do servidor (post_max_size).';
            return [$newPhotoPath, $errors];
        }

        if ($size <= 0) {
            $errors[] = 'Arquivo de foto ausente ou vazio.';
            return [$newPhotoPath, $errors];
        }
        if ($size > min($uplMax, 5 * 1024 * 1024)) {
            $errors[] = 'A imagem excede o tamanho máximo permitido (5MB).';
        }
        if (!in_array($ext, ['jpg','jpeg','png','webp'], true)) {
            $errors[] = 'Formato inválido. Use JPG, JPEG, PNG ou WEBP.';
        }
        if (!is_uploaded_file($tmp)) {
            $errors[] = 'Upload inválido.';
        } else {
            $finfo = finfo_open(FILEINFO_MIME_TYPE);
            $mime  = finfo_file($finfo, $tmp);
            finfo_close($finfo);
            if (strpos((string)$mime, 'image/') !== 0) {
                $errors[] = 'O arquivo selecionado não é uma imagem válida.';
            }
        }

        if ($errors) {
            return [$newPhotoPath, $errors];
        }

        // Pasta de destino
        $destDir = $this->publicRoot() . '/uploads/employees';
        if (!is_dir($destDir) && !mkdir($destDir, 0755, true)) {
            $errors[] = 'Não foi possível criar a pasta de uploads.';
            return [$newPhotoPath, $errors];
        }

        // Nome único
        $fileName = bin2hex(random_bytes(8)) . '.' . $ext;
        $destPath = $destDir . '/' . $fileName;

        if (!move_uploaded_file($tmp, $destPath)) {
            $errors[] = 'Falha ao salvar o upload no destino.';
            return [$newPhotoPath, $errors];
        }

        // Remove a foto anterior
        if (!empty($currentPhotoPath)) {
            $old = $this->publicRoot() . '/' . ltrim($currentPhotoPath, '/');
            if (is_file($old)) @unlink($old);
        }

        $newPhotoPath = 'uploads/employees/' . $fileName;
        return [$newPhotoPath, $errors];
    }

    private function photoUrl(?string $photo_path): string {
        $placeholder = BASE_URL . '/assets/img/placeholder-user.png';
        if (!$photo_path) return $placeholder;
        $fs = $this->publicRoot() . '/' . ltrim($photo_path, '/');
        return is_file($fs) ? (BASE_URL . '/' . ltrim($photo_path, '/')) : $placeholder;
    }

    /* ======================== AÇÕES ======================== */

    public function index() {
        AuthHelper::check();
        try {
            $filters = [];
            if (!empty($_GET['search'])) $filters['name'] = $_GET['search'];
            if (!empty($_GET['status'])) $filters['status'] = $_GET['status'];

            $rows = !empty($filters) ? $this->employeeModel->search($filters) : $this->employeeModel->getAll();
            require_once __DIR__ . '/../views/employees/colaboradores.php';
        } catch (Exception $e) {
            $_SESSION['error'] = 'Erro ao carregar colaboradores: ' . $e->getMessage();
            header('Location: ' . BASE_URL . '/rh'); exit;
        }
    }

    public function show($id) {
        AuthHelper::check();
        try {
            $employee = $this->employeeModel->findById((int)$id);
            if (!$employee) { $_SESSION['error'] = 'Colaborador não encontrado'; header('Location: ' . BASE_URL . '/employees'); exit; }
            // anexa URL da foto para a view (sem alterar banco)
            $employee['_photo_url'] = $this->photoUrl($employee['photo_path'] ?? null);
            require_once __DIR__ . '/../views/employees/show.php';
        } catch (Exception $e) {
            $_SESSION['error'] = 'Erro ao carregar colaborador: ' . $e->getMessage();
            header('Location: ' . BASE_URL . '/employees'); exit;
        }
    }

    public function create() {
        AuthHelper::check();
        require_once __DIR__ . '/../views/employees/create.php';
    }

    public function edit($id) {
        AuthHelper::check();
        try {
            $employee = $this->employeeModel->findById((int)$id);
            if (!$employee) { $_SESSION['error'] = 'Colaborador não encontrado'; header('Location: ' . BASE_URL . '/employees'); exit; }
            $employee['_photo_url'] = $this->photoUrl($employee['photo_path'] ?? null);
            require_once __DIR__ . '/../views/employees/edit.php';
        } catch (Exception $e) {
            $_SESSION['error'] = 'Erro ao carregar colaborador: ' . $e->getMessage();
            header('Location: ' . BASE_URL . '/employees'); exit;
        }
    }

    public function store() {
        AuthHelper::check();
        if (($_SERVER['REQUEST_METHOD'] ?? 'GET') !== 'POST') { header('Location: ' . BASE_URL . '/employees/create'); exit; }

        try {
            // valida básicos
            $required = ['full_name','cpf','rg','birth_date','address','phone_number','job_title','salary','hire_date','bank_name','agency_number','account_number'];
            foreach ($required as $f) if (empty($_POST[$f])) throw new Exception("Campo {$f} é obrigatório");

            if (!$this->employeeModel->validateCpf($_POST['cpf'])) throw new Exception('CPF inválido');
            if ($this->employeeModel->cpfExists($_POST['cpf']))   throw new Exception('CPF já cadastrado');

            $salary = $this->brMoneyToDecimal($_POST['salary']);
            if ($salary <= 0) throw new Exception('Salário deve ser um valor positivo');

            // upload da foto (opcional)
            [$photoPath, $uploadErrors] = $this->handlePhotoUpload(null, false);
            if ($uploadErrors) {
                $_SESSION['errors'] = $uploadErrors;
                $_SESSION['old']    = $_POST;
                header('Location: ' . BASE_URL . '/employees/create'); exit;
            }

            $data = [
                'full_name'      => trim($_POST['full_name']),
                'cpf'            => $_POST['cpf'],
                'rg'             => trim($_POST['rg']),
                'birth_date'     => $_POST['birth_date'],
                'address'        => trim($_POST['address']),
                'phone_number'   => trim($_POST['phone_number']),
                'job_title'      => trim($_POST['job_title']),
                'salary'         => $salary,
                'hire_date'      => $_POST['hire_date'],
                'contract_type'  => $_POST['contract_type'] ?? 'CLT',
                'cost_center'    => trim($_POST['cost_center'] ?? '') ?: null,
                'bank_name'      => trim($_POST['bank_name']),
                'agency_number'  => trim($_POST['agency_number']),
                'account_number' => trim($_POST['account_number']),
                'status'         => $_POST['status'] ?? 'ativo',
                'photo_path'     => $photoPath,
            ];

            $newId = $this->employeeModel->create($data);
            if (!$newId) throw new Exception('Erro ao cadastrar colaborador');

            $_SESSION['success'] = 'Colaborador cadastrado com sucesso.';
            header('Location: ' . BASE_URL . '/employees/' . (int)$newId); exit;

        } catch (Exception $e) {
            $_SESSION['error'] = $e->getMessage();
            $_SESSION['old']   = $_POST;
            header('Location: ' . BASE_URL . '/employees/create'); exit;
        }
    }

    public function update($id) {
        AuthHelper::check();
        if (($_SERVER['REQUEST_METHOD'] ?? 'GET') !== 'POST') { header('Location: ' . BASE_URL . '/employees/' . (int)$id . '/edit'); exit; }

        try {
            $employee = $this->employeeModel->findById((int)$id);
            if (!$employee) throw new Exception('Colaborador não encontrado');

            $required = ['full_name','cpf','rg','birth_date','address','phone_number','job_title','salary','hire_date','bank_name','agency_number','account_number'];
            foreach ($required as $f) if (empty($_POST[$f])) throw new Exception("Campo {$f} é obrigatório");

            if (!$this->employeeModel->validateCpf($_POST['cpf'])) throw new Exception('CPF inválido');
            if ($this->employeeModel->cpfExists($_POST['cpf'], (int)$id)) throw new Exception('CPF já cadastrado para outro colaborador');

            $salary = $this->brMoneyToDecimal($_POST['salary']);
            if ($salary <= 0) throw new Exception('Salário deve ser um valor positivo');

            $currentPhoto = $_POST['current_photo_path'] ?? ($employee['photo_path'] ?? null);
            $removePhoto  = !empty($_POST['remove_photo']);

            [$photoPath, $uploadErrors] = $this->handlePhotoUpload($currentPhoto, $removePhoto);
            if ($uploadErrors) {
                $_SESSION['errors'] = $uploadErrors;
                $_SESSION['old']    = $_POST;
                header('Location: ' . BASE_URL . '/employees/' . (int)$id . '/edit'); exit;
            }

            $data = [
                'full_name'      => trim($_POST['full_name']),
                'cpf'            => $_POST['cpf'],
                'rg'             => trim($_POST['rg']),
                'birth_date'     => $_POST['birth_date'],
                'address'        => trim($_POST['address']),
                'phone_number'   => trim($_POST['phone_number']),
                'job_title'      => trim($_POST['job_title']),
                'salary'         => $salary,
                'hire_date'      => $_POST['hire_date'],
                'contract_type'  => $_POST['contract_type'] ?? 'CLT',
                'cost_center'    => trim($_POST['cost_center'] ?? '') ?: null,
                'bank_name'      => trim($_POST['bank_name']),
                'agency_number'  => trim($_POST['agency_number']),
                'account_number' => trim($_POST['account_number']),
                'status'         => $_POST['status'] ?? 'ativo',
                'photo_path'     => $photoPath,
            ];

            $ok = $this->employeeModel->update((int)$id, $data);
            if (!$ok) throw new Exception('Erro ao atualizar colaborador');

            $_SESSION['success'] = 'Alterações salvas com sucesso.';
            header('Location: ' . BASE_URL . '/employees/' . (int)$id); exit;

        } catch (Exception $e) {
            $_SESSION['error'] = $e->getMessage();
            $_SESSION['old']   = $_POST;
            header('Location: ' . BASE_URL . '/employees/' . (int)$id . '/edit'); exit;
        }
    }

    public function destroy($id) {
        AuthHelper::check();
        try {
            $employee = $this->employeeModel->findById((int)$id);
            if (!$employee) throw new Exception('Colaborador não encontrado');

            if (!empty($employee['photo_path'])) {
                $old = $this->publicRoot() . '/' . ltrim($employee['photo_path'], '/');
                if (is_file($old)) @unlink($old);
            }

            $ok = $this->employeeModel->delete((int)$id);
            $_SESSION['success'] = $ok ? 'Colaborador excluído com sucesso.' : 'Erro ao excluir colaborador.';
        } catch (Exception $e) {
            $_SESSION['error'] = $e->getMessage();
        }
        header('Location: ' . BASE_URL . '/employees'); exit;
    }

    /* Métodos em lote / documentos — mantenha os seus se já usa */
}
